import json
import os
import uuid
from flask import Flask, redirect, render_template, request, session
from flask_session import Session
from glob import glob
from pathlib import Path
from datetime import datetime

app = Flask(__name__)

# NEW: Configure session
app.config["SESSION_PERMANENT"] = False
app.config["SESSION_TYPE"] = "filesystem"
Session(app)

# Ensure data directory exists
DATA_DIR = Path("data")
DATA_DIR.mkdir(exist_ok=True)

SPORTS = [
    "Basketball",
    "Soccer",
    "Ultimate Frisbee"
    ]

@app.route("/")
def index():
    return render_template("index.html", sports=SPORTS,
                           name=session.get("username"))

# NEW
@app.route("/login", methods=["GET", "POST"])
def login():
    if request.method == "POST":
        # Note: In a real application, you would verify
        # passwords here. Do not implement your own
        # authentication system. Use established libraries
        # and follow security best practices!
        session["username"] = request.form.get("name")
        session["history"] = []  # used to keep track of user's activities
        return redirect("/")
    return render_template("login.html")


# NEW: forget the user's session to log them out
@app.route("/logout")
def logout():
    session.clear()
    return redirect("/")


@app.route("/register", methods=["POST"])
def register():
    # NEW: Ensure user is logged in
    if not session.get("username"):
        return redirect("/login")

    # Validate name
    name = request.form.get("name")
    if not name:
        return render_template("error.html", message="Missing name")

    # Validate sports
    sports = request.form.getlist("sport")
    if not sports:
        return render_template("error.html", message="Missing sport")
    for sport in sports:
        if sport not in SPORTS:
            return render_template("error.html", message="Invalid sport")

    # Create registrant with UUID
    registrant = {
        "id": str(uuid.uuid4()),
        "username": session["username"],  # NEW
        "name": name,
        "sports": sports
    }

    # Save to JSON file
    with open(DATA_DIR / f"{registrant['id']}.json", "w") as f:
        json.dump(registrant, f)

    # NEW: Keep track of user's activities
    session["history"].append({
        "time": datetime.now().strftime("%H:%M:%S"),
        "action": "You registered for " + ", ".join(sports)
    })

    return redirect("/registrants")


@app.route("/deregister", methods=["POST"])
def deregister():
    # NEW: Ensure user is logged in
    if not session.get("username"):
        return redirect("/login")

    id = request.form.get("id")
    sport = request.form.get("sport")
    if id:
        file_path = DATA_DIR / f"{id}.json"
        if file_path.exists():
            with open(file_path) as f:
                reg = json.load(f)
                # NEW: check that the id received from the
                # browser (hidden form field) actually matches
                # the currently logged-in user. If not, the
                # user tampered with the id and intends to
                # modify a file of someone else. That is not
                # allowed!
                if reg["username"] == session["username"]:
                    reg["sports"].remove(sport)
                    with open(file_path, "w") as g:
                        json.dump(reg, g)
                else:
                    return render_template("error.html", message="Not authorized")

    # NEW: again, keep track of their activities
    session["history"].append({
        "time": datetime.now().strftime("%H:%M:%S"),
        "action": f"You de-registered from {sport}"
    })

    return redirect("/registrants")


@app.route("/registrants")
def registrants():
    # NEW: Ensure user is logged in
    if not session.get("username"):
        return redirect("/login")

    # Collect the user's registrations
    flat_registrants = []
    file_id = ""
    for file_path in DATA_DIR.glob("*.json"):
        with open(file_path) as f:
            reg = json.load(f)
            # NEW: Only show user's registrations
            if reg["username"] == session["username"]:
                # Flatten sports list for template compatibility
                for sport in reg["sports"]:
                    flat_registrants.append({
                        "id": reg["id"],
                        "name": reg["name"],
                        "sport": sport
                    })

    return render_template("registrants.html", registrants=flat_registrants)
